/*
 * Copyright (C) 2004 Josip Deanovic <djosip@linuxpages.org>
 * This program is copyrighted under GPL license. See COPYING file
 * for details.
 *
 * mod_sar is output filter for apache2. It's purpose is to search
 * and replace strings before it's sending to the client.
 * Compile: apxs -c mod_sar.c
 * Install: apxs -a -i -c mod_sar.c
 */



/* DEFINES */
#define VERSION "mod_sar/1.0"


/* INCLUDES */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <httpd.h>
#include <http_config.h>
#include <apr_strings.h>
#include <util_filter.h>


/* PROTOTYPES */
module AP_MODULE_DECLARE_DATA sar_module;


/* VARIABLES AND STRUCTURES */
typedef struct
{
    char *search_string;
    char *replace_string;
    int case_insensitive;
    int verbose;
} sar_config;

typedef struct
{
    apr_bucket_brigade *bb;
} sar_struct;



static void *
create_sar_config (apr_pool_t * p, char *dummy)
{
    /* Allocating memory from provided pool. */
    sar_config *cfg = (sar_config *) apr_pcalloc (p, sizeof (sar_config));

    /* Default config values */
    cfg->search_string = NULL;
    cfg->replace_string = NULL;
    cfg->case_insensitive = 0;
    cfg->verbose = 0;

    return (void *) cfg;
}



/* Callback function for SarStrings directive. */
static const char *
set_strings (cmd_parms * parms, void *dummy, const char *search_string,
	     const char *replace_string)
{
    sar_config *cfg = dummy;

    cfg->search_string = (char *) search_string;
    cfg->replace_string = (char *) replace_string;

    return NULL;
}



/* A handler function for HTTP methods */
static int
mod_sar_method_handler (request_rec * r)
{
    sar_config *cfg = ap_get_module_config (r->per_dir_config, &sar_module);

    if (cfg->search_string == NULL || cfg->replace_string == NULL)
      {
          fprintf (stderr, "mod_sar: warning: search string or replace string"
			   " found to be NULL\n");
          fflush (stderr);
      }

    if (cfg->verbose == 1 && cfg->search_string != NULL && cfg->replace_string != NULL)
      {
	  fprintf (stderr, "SEARCH: %s, REPLACE: %s, CASE: %d, VERBOSE: %d\n",
		   cfg->search_string, cfg->replace_string,
		   cfg->case_insensitive, cfg->verbose);
	  fflush (stderr);
      }

    return DECLINED;
}



/* Handler function which will inform server with version of this module. */
static int
mod_sar_version (apr_pool_t * p, apr_pool_t * dummy1, apr_pool_t * dummy2,
		 server_rec * dummy3)
{
    ap_add_version_component (p, VERSION);

    return OK;
}



/* Case insensitive strstr(3) replacement function. */
static char *
ncasestrstr (const char *source_string, const char *search_string)
{
    char *source_ptr;
    char *search_ptr = (char *) search_string;
    char *start_ptr = (char *) source_string;
    unsigned int source_len = strlen (source_string);
    unsigned int search_len = strlen (search_string);

    /* Loop until length of source string became shorter then search string. */
    for (; source_len >= search_len; start_ptr++, source_len--)
      {
	  /* Find the start of search string in the source string. */
	  while (tolower (*start_ptr) != tolower (*search_string))
	    {
		start_ptr++;
		source_len--;

		/* Search string cannot be longer than source string. */
		if (search_len > source_len)
		    return (NULL);
	    }

	  source_ptr = start_ptr;
	  search_ptr = (char *) search_string;

	  while (tolower (*source_ptr) == tolower (*search_ptr))
	    {
		source_ptr++;
		search_ptr++;

		/* If true, then search string was found. */
		if (*search_ptr == '\0')
		    return (start_ptr);
	    }
      }
    return (NULL);
}



/* Search and replace function. */
static char *
replace (apr_pool_t * p, sar_config * cfg, const char *source_string,
         const char *search_string, const char *replace_string)
{
    size_t source_size = strlen (source_string) + 1;
    size_t search_size = strlen (search_string);
    size_t replace_size = strlen (replace_string);
    char *value = apr_pcalloc (p, source_size);
    char *dst = value;

    if (value != NULL)
      {
	  /* Loop until no matches are found. */
	  for (;;)
	    {
		const char *match;
		/* Try to find the search string. */
		if (cfg->case_insensitive == 1)
		    match = ncasestrstr (source_string, search_string);
		else
		    match = strstr (source_string, search_string);

		if (match != NULL)
		  {
		      /* Find out how many characters to copy up to the 'match'. */
		      size_t count = match - source_string;
		      /* We will need a temporary pointer for safe value realloc. */
		      char *temp = apr_pcalloc (p, source_size);
		      /* Calculate the total size the string will be after the replacement. */
		      source_size += replace_size - search_size;
		      temp = value;
		      value = apr_pcalloc (p, source_size);
		      value = temp;
		      /*
		       * Copy from the source to the point where we matched. Then
		       * move the source pointer ahead by the amount we copied. And
		       * move the destination pointer ahead by the same amount.
		       */
		      memmove (dst, source_string, count);
		      source_string += count;
		      dst += count;
		      /*
		       * Copy the replace string at the position of the match.
		       * Adjust the source pointer by the text we replaced.
		       * Adjust the destination pointer by the amount of replacement
		       * text.
		       */
		      memmove (dst, replace_string, replace_size);
		      source_string += search_size;
		      dst += replace_size;
		  }
		else /* No match found. */
		  {
		      /* Copy remaining part of the string, including '0'. */
		      strcpy (dst, source_string);
		      break;
		  }
	    } /* Endless loop. */
      }

    return value;
}



/* Output filter fuction. */
static int
sar_filter (ap_filter_t * f, apr_bucket_brigade * bb)
{
    request_rec *r = f->r;
    sar_config *cfg = ap_get_module_config (r->per_dir_config, &sar_module);
    sar_struct *ctx = f->ctx;
    apr_bucket *e;
    apr_pool_t *p = f->r->pool;

    if (ctx == NULL)
      {
	  f->ctx = ctx = apr_pcalloc (f->r->pool, sizeof (*ctx));
	  ctx->bb = apr_brigade_create (f->r->pool, f->c->bucket_alloc);
      }

    APR_BRIGADE_FOREACH (e, bb)
    {
	const char *str;
	apr_size_t len;
	char *after;

	if (APR_BUCKET_IS_EOS (e) || APR_BUCKET_IS_FLUSH (e))
	  {
	      APR_BUCKET_REMOVE (e);
	      APR_BRIGADE_INSERT_TAIL (ctx->bb, e);
	      ap_pass_brigade (f->next, ctx->bb);
	      return APR_SUCCESS;
	  }

	apr_bucket_read (e, &str, &len, APR_NONBLOCK_READ);

        /* Checking if search and replace string exists. */
        if (cfg->search_string != NULL && cfg->replace_string != NULL)
	  if (NULL != (after = replace (f->r->pool, cfg, str, cfg->search_string,
		       cfg->replace_string)))
	      str = after;

	ap_fputs (f->next, ctx->bb, str);
    }

    return APR_SUCCESS;
}



/* A callback function which declares handlers for other events. */
static void
sar_register_hooks (apr_pool_t * p)
{
    ap_hook_handler (mod_sar_method_handler, NULL, NULL, APR_HOOK_LAST);
    ap_register_output_filter ("sar_filter", sar_filter, NULL,
			       AP_FTYPE_CONTENT_SET);
    ap_hook_post_config (mod_sar_version, NULL, NULL, APR_HOOK_MIDDLE);
}



/* Declaration of the directives supported by mod_sar. */
static const command_rec sar_cmds[] =
{
    AP_INIT_TAKE2 ("SarStrings", set_strings,
		   (void *) APR_OFFSETOF (sar_config, search_string),
		   RSRC_CONF, "search string and replace string."),
    AP_INIT_TAKE2 ("SarStrings", set_strings,
		   (void *) APR_OFFSETOF (sar_config, replace_string),
		   RSRC_CONF, "search string and replace string."),
    AP_INIT_FLAG ("SarCaseInsensitive", ap_set_flag_slot,
		  (void *) APR_OFFSETOF (sar_config, case_insensitive),
		  RSRC_CONF,
		  "SarCaseInsensitive accepts either On or Off flag."),
    AP_INIT_FLAG ("SarVerbose", ap_set_flag_slot,
		  (void *) APR_OFFSETOF (sar_config, verbose), RSRC_CONF,
		  "SarVerbose accepts either On or Off flag."),
    {NULL}
};



/* Module's data structure. */
module AP_MODULE_DECLARE_DATA sar_module =
{
    STANDARD20_MODULE_STUFF,
    create_sar_config,
    NULL,
    NULL,
    NULL,
    sar_cmds,
    sar_register_hooks,
};
