/*
 * Copyright (C) 2004 Josip Deanovic <djosip@linuxpages.org>
 * This program is copyrighted under GPL license. See COPYING file
 * for details.
 */



/* INCLUDES */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <time.h>
#include <string.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include <X11/Xresource.h>
#include <X11/Xmu/Error.h>
#include <X11/extensions/scrnsaver.h>

#ifdef GETOPT_LONG
#include <getopt.h>
#endif



/* DEFINES */
#define VERSION "1.1"
#define PROGNAME "rsaver"



/* GLOBAL VARIABLES */
Display *display;		/* Our display */
int screen;			/* Our screen number */
unsigned int screen_width;	/* Screen width */
unsigned int screen_height;	/* Screen height */
Window root_window;		/* Our root window */
GC gc;				/* Default Graphics Context */
GC bgc;				/* Graphics Context with black foreground */
Colormap colormap;		/* Colormap structure */
XColor color;			/* Used for "named colors" */
XColor truecolor;		/* Used for truecolors (RGB colors) */
Cursor cursor;			/* Used for "stealth" pointer */
Pixmap cursor_mask;		/* Pixmap for cursor mask */
Pixmap buffer;			/* Pixmap over whole screen used for buffering */
Pixmap small_buffer;		/* Pixmap used for buffering */
XScreenSaverInfo *ssinfo;	/* Used for informations from screensaver ext. */
XColor dummy;			/* Unused color structure */
XSetWindowAttributes attrs;	/* Structure used for window attributes */
XGCValues gcvalues;		/* GC values, used for bgc */
XEvent event;			/* XEvent structure */
char user_color[32] = "yellow";	/* Used for user specified color */



/* PROTOTYPES */
void help (void);
void radioactive (int x, int y, int start_angle);
void clear_screen (int side);
void colorand (void);
void set_color (char *new_color);
int errorhandler (Display * display, XErrorEvent * errorp);
void sighandler (int signum);



/* MAIN FUNCTION */
int
main (int argc, char *argv[])
{
/* Variables needed in main function only */
    char *disp = NULL;		/* User specified display */
    int parameter = 0;		/* Used for return from getopt and getopt_long */
    char *option;		/* Used for return from XGetDefault */
    int set_cw = 0;		/* Used for drawing in cw or ccw direction */
    int timeout = 0;		/* Time of screensaver inactivity */
    int set_timeout = 0;	/* User specified timeout */
    int random_colors = 0;	/* If user want random colors */
    int amount = 16;		/* Amount of objects on the screen */
    int amount_set = 0;		/* This indicates whether amount has been set by user */
    int noshaped = 0;		/* Do not show shaped radioactive signs */
    Visual *visual;		/* Our visual */
    unsigned int depth;		/* Screen depth */
    Pixmap pix_blank;		/* Used for creating cursor pixmap */
    int event_base = 0;		/* Needed for functions from XScreenSaver ext. */
    int error_base = 0;		/* Needed for functions from XScreenSaver ext. */
    int count;			/* Counter */
    int count2;			/* Counter */
    int x;			/* X part of radioactive sign coordinates */
    int y;			/* Y part of radioactive sign coordinates */
    unsigned long old_idle;	/* Buffer used to save ssinfo->idle */


/*
 * Getting arguments using getopt or getopt_long.
 * On systems without getopt_long, getopt will be used instead.
 */
    while (!0)
      {
#ifdef GETOPT_LONG
	  int option_index = 0;
	  static struct option long_options[] = {
	      {"cw", 0, 0, 0},
	      {"display", 1, 0, (int) 'd'},
	      {"timeout", 1, 0, (int) 't'},
	      {"color", 1, 0, (int) 'c'},
	      {"amount", 1, 0, (int) 'a'},
	      {"noshaped", 0, 0, (int) 'n'},
	      {"version", 0, 0, (int) 'V'},
	      {"help", 0, 0, (int) 'h'},
	      {0, 0, 0, 0}
	  };

	  parameter = getopt_long (argc, argv, "d:t:c:a:nVh", long_options,
				   &option_index);
#else
	  parameter = getopt (argc, argv, "wd:t:c:a:nVh");
#endif /* GETOPT_LONG */
	  if (parameter == -1)
	      break;

	  switch (parameter)
	    {
#ifdef GETOPT_LONG
	    case 0:
		if (strcmp (long_options[option_index].name, "cw") == 0)
		    set_cw = 1;
		break;
#else
	    case 'w':
		set_cw = 1;
		break;
#endif /* GETOPT_LONG */
	    case 'd':
		disp = optarg;
		break;
	    case 't':
		timeout = atoi (optarg);
		if (timeout <= 0)
		    timeout = 1;
		set_timeout = 1;
		break;
	    case 'c':
		if (strlen (optarg) >= 30)
		  {
		      (void) fprintf (stderr, "Unknown color\n");
		      exit (EXIT_FAILURE);
		  }
		else if (strcmp (optarg, "random") == 0)
		  {
		      random_colors = 1;
		      colorand ();
		  }
		else if (strcmp (optarg, "multi") == 0)
		    random_colors = 2;
		else
		  {
		      strcpy (user_color, optarg);
		      random_colors = 3;
		  }
		break;
	    case 'a':
		amount = atoi (optarg);
		if (amount < 0)
		    amount = 0;
		amount_set = 1;
		break;
	    case 'n':
		noshaped = 1;
		break;
	    case 'V':
		(void) fprintf (stderr, "%s %s, (C) 2004 Josip Deanovic "
				"<djosip@linuxpages.org>\n", PROGNAME,
				VERSION);
		exit (EXIT_SUCCESS);
	    case 'h':
		help ();
		exit (EXIT_SUCCESS);
	    default:
		help ();
		break;
	    }
      }

    if (optind < argc)
      {
	  help ();
	  exit (EXIT_FAILURE);
      }


/* Connecting to a server */
    if ((display = XOpenDisplay (disp)) == NULL)
      {
	  (void) fprintf (stderr, "%s: cannot connect to X server %s\n",
			  PROGNAME, XDisplayName (disp));
	  exit (EXIT_FAILURE);
      }


/* Getting options not set with command line arguments from X resource db. */
    if (set_timeout == 0)
	if ((option = XGetDefault (display, PROGNAME, "timeout")))
	  {
	      timeout = atoi (option);
	      if (timeout <= 0)
		  timeout = 1;
	      set_timeout = 1;
	  }

    if (random_colors == 0)
      {
	  if ((option = XGetDefault (display, PROGNAME, "color")))
	    {
		if (strlen (option) >= 30)
		  {
		      (void) fprintf (stderr, "Unknown color\n");
		      exit (EXIT_FAILURE);
		  }
		else if (strcmp (option, "random") == 0)
		    random_colors = 1;
		else if (strcmp (option, "multi") == 0)
		    random_colors = 2;
		else
		  {
		      strcpy (user_color, option);
		      random_colors = 3;
		  }
	    }
      }

    if (set_cw == 0)
	if ((option = XGetDefault (display, PROGNAME, "cw")))
	    if (strcmp (option, "on") == 0)
		set_cw = 1;

    if (amount_set == 0)
	if ((option = XGetDefault (display, PROGNAME, "amount")))
	  {
	      amount = atoi (option);
	      amount_set = 1;
	  }

    if (noshaped == 0)
	if ((option = XGetDefault (display, PROGNAME, "noshaped")))
	    if (strcmp (option, "on") == 0)
		noshaped = 1;


/* Using macros provided by Xlib to obtain info about display. */
    screen = DefaultScreen (display);
    screen_width = (unsigned int) DisplayWidth (display, screen);
    screen_height = (unsigned int) DisplayHeight (display, screen);
    depth = (unsigned int) DefaultDepth (display, screen);
    visual = DefaultVisual (display, screen);
    colormap = XDefaultColormap (display, screen);
    root_window = DefaultRootWindow (display);


/* Testing the existence of Xscreensaver extension. */
    if (XScreenSaverQueryExtension (display, &event_base, &error_base) ==
	False)
      {
	  (void) fprintf (stderr,
			  "%s: XScreenSaverQueryExtension - screensaver "
			  "extension not supported\n", PROGNAME);
	  exit (EXIT_FAILURE);
      }

/* Allocating an XScreenSaverInfo structure. */
    if ((ssinfo = XScreenSaverAllocInfo ()) == NULL)
      {
	  (void) fprintf (stderr,
			  "%s: XScreenSaverAllocInfo - insufficient memory\n",
			  PROGNAME);
	  exit (EXIT_FAILURE);
      }

/*
 * If some other client already used XScreenSaverSetAttributes we should not
 * try to do the same. It means that external screensaver will be used and
 * ssinfo->kind will be ScreenSaverExternal.
 */
    if (XScreenSaverQueryInfo (display, root_window, ssinfo) == 0)
      {
	  (void) fprintf (stderr, "%s: XScreenSaverQueryInfo - extension not "
			  "supported\n", PROGNAME);
	  exit (EXIT_FAILURE);
      }

    if (ssinfo->kind == ScreenSaverExternal)
      {
	  (void) fprintf (stderr, "%s: some other client has already set the "
			  "screensaver attributes.\n", PROGNAME);
	  exit (EXIT_FAILURE);
      }

/* Create _MIT_SCREEN_SAVER_ID property on the root window. */
    if (XScreenSaverRegister (display, screen, ssinfo->window, XA_WINDOW) ==
	0)
	(void) fprintf (stderr,
			"%s: XScreenSaverRegister - unable to register\n",
			PROGNAME);


/*
 * Catch some signals upon whose arrival client sighandler will be called.
 * This is needed because client have to remove _MIT_SCREEN_SAVER_ID property
 * from root window before exiting.
 */
    (void) signal (SIGHUP, sighandler);
    (void) signal (SIGINT, sighandler);
    (void) signal (SIGPIPE, sighandler);
    (void) signal (SIGALRM, sighandler);
    (void) signal (SIGTERM, sighandler);


/* Creating invisible cursor. */
    if (XAllocNamedColor (display, colormap, "black", &dummy, &dummy) == 0)
      {
	  (void) fprintf (stderr,
			  "%s: XAllocNamedColor - insufficient memory\n",
			  PROGNAME);
	  exit (EXIT_FAILURE);
      }

    pix_blank = XCreatePixmap (display, root_window, 1, 1, 1);
    if ((attrs.cursor = XCreatePixmapCursor (display, pix_blank, pix_blank,
					     &dummy, &dummy, 0, 0)) == 0)
      {
	  (void) fprintf (stderr,
			  "%s: XCreatePixmapCursor - insufficient memory\n",
			  PROGNAME);
	  exit (EXIT_FAILURE);
      }

    (void) XFreePixmap (display, pix_blank);

/* Other window (screensaver) attributes. */
    attrs.background_pixel = BlackPixel (display, screen);
    attrs.border_pixel = BlackPixel (display, screen);

/*
 * Normally, we would use window with override_redirect attribute set
 * but we are using screensaver extension.
 */
    XScreenSaverSetAttributes (display, root_window, 0, 0, screen_width,
			       screen_height, 0, (int) depth, InputOutput,
			       visual, (unsigned long) CWBackPixel |
			       CWBorderPixel | CWCursor, &attrs);


/* Creating pixmaps for double buffering. */
    if ((buffer = XCreatePixmap (display, root_window, screen_width,
				 screen_height, depth)) == 0)
      {
	  (void) fprintf (stderr, "%s: XCreatePixmap - insufficient memory\n",
			  PROGNAME);
	  exit (EXIT_FAILURE);
      }

    if ((small_buffer = XCreatePixmap (display, root_window, 125, 125,
				       depth)) == 0)
      {
	  (void) fprintf (stderr, "%s: XCreatePixmap - insufficient memory\n",
			  PROGNAME);
	  exit (EXIT_FAILURE);
      }


/* Getting the "Graphics Context" for the root window. */
    if ((gc = DefaultGC (display, screen)) == NULL)
      {
	  (void) fprintf (stderr, "%s: could not find a graphics context\n",
			  PROGNAME);
	  exit (EXIT_FAILURE);
      }
    (void) XSetGraphicsExposures (display, gc, False);

/* Creating a Graphics Context with black foreground. */
    gcvalues.foreground = BlackPixel (display, screen);
    gcvalues.graphics_exposures = False;

    if ((bgc = XCreateGC (display, root_window, (unsigned long) GCForeground |
			  GCGraphicsExposures, &gcvalues)) == NULL)
      {
	  (void) fprintf (stderr, "%s: could not create a graphics context\n",
			  PROGNAME);
	  exit (EXIT_FAILURE);
      }


/* If user did not specify color or random color, yellow color will be used. */
    if (random_colors != 2)
	set_color (user_color);


/* Selecting wanted events. */
    XScreenSaverSelectInput (display, root_window, ScreenSaverNotifyMask);


/* Event loop */
    while (!0)
      {
	  /*
	   * If user did not specify timeout, rsaver will start drawing when
	   * screen saver activates.
	   */
	  if (set_timeout == 0)
	    {
		(void) XSync (display, True);
		(void) XNextEvent (display, &event);

		if (event.type == ScreenSaverNotify
		    && event.type == event_base)
		  {
#ifdef USE_DEBUG
		      (void) fprintf (stderr, "%s: unknown event: %d\n",
				      PROGNAME, event.type);
#endif
		      continue;
		  }
	    }


	  if (XScreenSaverQueryInfo (display, root_window, ssinfo) == 0)
	    {
		(void) fprintf (stderr,
				"%s: XScreenSaverQueryInfo - extension not "
				"supported\n", PROGNAME);
		exit (EXIT_FAILURE);
	    }

#ifdef USE_DEBUG
	  fprintf (stderr,
		   "%s: state: %d kind: %d til_or_since: %ld idle: %ld "
		   "timeout: %d\n", PROGNAME, ssinfo->state, ssinfo->kind,
		   ssinfo->til_or_since, ssinfo->idle, timeout);
#endif


	  if (ssinfo->idle / 1000 >= (unsigned int) timeout)
	    {
		if (ssinfo->state != ScreenSaverOn)
		    (void) XForceScreenSaver (display, ScreenSaverActive);

		/* Pixmap should be cleared before we use it. */
		(void) XFillRectangle (display, buffer, bgc, 0, 0,
				       screen_width, screen_height);


		if (XScreenSaverQueryInfo (display, root_window, ssinfo) == 0)
		  {
		      (void) fprintf (stderr,
				      "%s: XScreenSaverQueryInfo - extension not "
				      "supported\n", PROGNAME);
		      exit (EXIT_FAILURE);
		  }

		old_idle = ssinfo->idle;

		/* Rotation direction. */
		if (set_cw == 0)
		    count = 0;
		else
		    count = 120;

		/*
		 * If user specified color as random, the color will change every
		 * time screen saver activates.
		 */
		if (random_colors == 1)
		  {
		      colorand ();
		      set_color (user_color);
		  }

		for (count2 = 0;;)
		  {
		      /*
		       * If user specified color as multi, the color will change for
		       * every radioactive sign.
		       */
		      if (random_colors == 2)
			{
			    colorand ();
			    set_color (user_color);
			}

		      if (set_cw == 0)
			{
			    count++;

			    if (count >= 120)
			      {
				  count = 0;
				  count2++;
			      }
			}
		      else
			{
			    count--;

			    if (count <= 0)
			      {
				  count = 120;
				  count2++;
			      }
			}

		      srandom ((unsigned long) time (NULL));

		      x = random () % ((int) screen_width + 1);
		      y = random () % ((int) screen_height + 1);

		      if (noshaped == 0 && (random () % 2) == 0)
			{
			    (void) XSetArcMode (display, gc, ArcChord);
			    (void) XSetArcMode (display, bgc, ArcChord);
			}
		      else
			{
			    (void) XSetArcMode (display, gc, ArcPieSlice);
			    (void) XSetArcMode (display, bgc, ArcPieSlice);
			}

		      if (count2 >= amount)
			{
			    count2 = 0;
			    srandom ((unsigned long) time (NULL));
			    clear_screen (random () % 4);

			    /* The color of line that clears the screen will change. */
			    if (random_colors == 1)
			      {
				  colorand ();
				  set_color (user_color);
			      }

			    if (old_idle >= ssinfo->idle)
				break;
			    else
				old_idle = ssinfo->idle;
			}
		      else
			{
			    /* Drawing radioactive sign at random positions. */
			    srandom ((unsigned long) time (NULL));

			    if (set_cw == 0)
				for (; count <= 120; count++)
				  {
				      radioactive (x, y, count);

				      (void) usleep (10000);

				      if (old_idle >= ssinfo->idle)
					  break;
				      else
					  old_idle = ssinfo->idle;
				  }
			    else
				for (; count >= 0; count--)
				  {
				      radioactive (x, y, count);

				      (void) usleep (10000);

				      if (old_idle >= ssinfo->idle)
					  break;
				      else
					  old_idle = ssinfo->idle;
				  }

			    if (old_idle > ssinfo->idle)
				break;
			    else
				old_idle = ssinfo->idle;
			}
		  }
		(void) XFlush (display);
	    }
	  (void) sleep (1);
      }


/* This point should never be reached but we will close display anyway. */
    (void) XCloseDisplay (display);

/* This will return 0 on most platforms. */
    return EXIT_SUCCESS;
}



/* FUNCTIONS */

/*
 * Function that prints the help screen.
 */
void
help (void)
{
#ifdef GETOPT_LONG
    (void) fprintf (stderr,
		    "Radioactive screensaver %s, (C) 2004 Josip Deanovic "
		    "<djosip@linuxpages.org>\n" "Usage: %s [options]\n"
		    "\t-d, --display <display>\t\tdisplay\n"
		    "\t-t, --timeout <seconds>\t\ttime of user inactivity\n"
		    "\t-c, --color <color>\t\tcolor (named color, random, multi)\n"
		    "\t    --cw\t\t\tclock wise\n"
		    "\t-a, --amount <amount>\t\tamount of objects shown\n"
		    "\t-n, --noshaped\t\t\tdo not show shaped radioactive signs\n"
		    "\t-V, --version\t\t\tshow version and exit\n"
		    "\t-h, --help\t\t\tshow this help screen\n", VERSION,
		    PROGNAME);
#else
    (void) fprintf (stderr,
		    "Radioactive screensaver %s, (C) 2004 Josip Deanovic "
		    "<djosip@linuxpages.org>\n" "Usage: %s [options]\n"
		    "\t-d <display>\t\tdisplay\n"
		    "\t-t <seconds>\t\ttime of user inactivity\n"
		    "\t-c <color>\t\tcolor (named color, random, multi)\n"
		    "\t-w\t\t\tclock wise\n"
		    "\t-a <amount>\t\tamount of objects shown\n"
		    "\t-n\t\t\tdo not show shaped radioactive signs\n"
		    "\t-V\t\t\tshow version and exit\n"
		    "\t-h\t\t\tshow this help screen\n", VERSION, PROGNAME);
#endif
    exit (EXIT_SUCCESS);
}



/*
 * Draws radioactive sign (by using XDrawArc) at random location.
 */
void
radioactive (int x, int y, int start_angle)
{
    (void) XCopyArea (display, buffer, small_buffer, gc, (x - (124 / 2)),
		      (y - (124 / 2)), 125, 125, 0, 0);

    (void) XDrawArc (display, small_buffer, gc, 0, 0, 124, 124, 0, 360 * 64);
    (void) XFillArc (display, small_buffer, gc, 4, 4, 116, 116,
		     (0 + start_angle) * 64, 60 * 64);
    (void) XFillArc (display, small_buffer, bgc, 4, 4, 116, 116,
		     (60 + start_angle) * 64, 60 * 64);
    (void) XFillArc (display, small_buffer, gc, 4, 4, 116, 116,
		     (120 + start_angle) * 64, 60 * 64);
    (void) XFillArc (display, small_buffer, bgc, 4, 4, 116, 116,
		     (180 + start_angle) * 64, 60 * 64);
    (void) XFillArc (display, small_buffer, gc, 4, 4, 116, 116,
		     (240 + start_angle) * 64, 60 * 64);
    (void) XFillArc (display, small_buffer, bgc, 4, 4, 116, 116,
		     (-60 + start_angle) * 64, 60 * 64);
    (void) XFillArc (display, small_buffer, bgc, 42, 42, 40, 40, 0, 360 * 64);
    (void) XFillArc (display, small_buffer, gc, 47, 47, 30, 30, 0, 360 * 64);

    if (XScreenSaverQueryInfo (display, root_window, ssinfo) == 0)
      {
	  (void) fprintf (stderr, "%s: XScreenSaverQueryInfo - extension not "
			  "supported\n", PROGNAME);
	  exit (EXIT_FAILURE);
      }

    if (ssinfo->state != ScreenSaverOn)
	return;
    else
      {
	  (void) XSetErrorHandler (errorhandler);
	  (void) XCopyArea (display, small_buffer, ssinfo->window, gc, 0, 0,
			    125, 125, (x - (124 / 2)), (y - (124 / 2)));
	  (void) XCopyArea (display, small_buffer, buffer, gc, 0, 0, 125, 125,
			    (x - (124 / 2)), (y - (124 / 2)));
	  (void) XSync (display, False);
	  (void) XSetErrorHandler (NULL);
      }

    (void) XFlush (display);
}



/*
 * This fuction can clear screen from top, bottom, left and the right side.
 * Function clear_screen expects one of these parameters:
 * 0: clear screen from top to the bottom
 * 1: clear screen from bottom to the top
 * 2: clear screen from left to the right
 * 3: clear screen from right to the left
 */
void
clear_screen (int side)
{
    int i;

    switch (side)
      {
      case 0:
	  i = 0;
	  (void) XDrawLine (display, buffer, bgc, 0, 0,
			    (int) screen_width, 0);
	  (void) XDrawLine (display, buffer, gc, 0, 1, (int) screen_width, 1);

	  while (i <= (int) screen_height)
	    {
		if (XScreenSaverQueryInfo (display, root_window, ssinfo) == 0)
		  {
		      (void) fprintf (stderr, "%s: XScreenSaverQueryInfo - "
				      "extension not supported\n", PROGNAME);
		      exit (EXIT_FAILURE);
		  }

		if (ssinfo->state != ScreenSaverOn)
		    return;
		else
		  {
		      (void) XSetErrorHandler (errorhandler);
		      (void) XCopyArea (display, buffer, ssinfo->window,
					gc, 0, 0, screen_width, 2, 0, i);
		      (void) XSetErrorHandler (NULL);
		  }
		(void) XFlush (display);
		(void) usleep (5000);
		i++;
	    }
	  break;
      case 1:
	  i = (int) screen_height;
	  (void) XDrawLine (display, buffer, gc, 0, 0, (int) screen_width, 0);
	  (void) XDrawLine (display, buffer, bgc, 0, 1,
			    (int) screen_width, 1);

	  while (i >= 0)
	    {
		if (XScreenSaverQueryInfo (display, root_window, ssinfo) == 0)
		  {
		      (void) fprintf (stderr, "%s: XScreenSaverQueryInfo - "
				      "extension not supported\n", PROGNAME);
		      exit (EXIT_FAILURE);
		  }

		if (ssinfo->state != ScreenSaverOn)
		    return;
		else
		  {
		      (void) XSetErrorHandler (errorhandler);
		      (void) XCopyArea (display, buffer, ssinfo->window,
					gc, 0, 0, screen_width, 2, 0,
					(i - 1));
		      (void) XSetErrorHandler (NULL);
		  }
		(void) XFlush (display);
		(void) usleep (5000);
		i--;
	    }
	  break;
      case 2:
	  i = 0;
	  (void) XDrawLine (display, buffer, bgc, 0, 0,
			    0, (int) screen_height);
	  (void) XDrawLine (display, buffer, gc, 1, 0,
			    1, (int) screen_height);

	  while (i <= (int) screen_width)
	    {
		if (XScreenSaverQueryInfo (display, root_window, ssinfo) == 0)
		  {
		      (void) fprintf (stderr, "%s: XScreenSaverQueryInfo - "
				      "extension not supported\n", PROGNAME);
		      exit (EXIT_FAILURE);
		  }

		if (ssinfo->state != ScreenSaverOn)
		    return;
		else
		  {
		      (void) XSetErrorHandler (errorhandler);
		      (void) XCopyArea (display, buffer, ssinfo->window, gc,
					0, 0, 2, screen_height, i, 0);
		      (void) XSetErrorHandler (NULL);
		  }
		(void) XFlush (display);
		(void) usleep (5000);
		i++;
	    }
	  break;
      case 3:
	  i = (int) screen_width;
	  (void) XDrawLine (display, buffer, gc, 0, 0,
			    0, (int) screen_height);
	  (void) XDrawLine (display, buffer, bgc, 1, 0,
			    1, (int) screen_height);

	  while (i >= 0)
	    {
		if (XScreenSaverQueryInfo (display, root_window, ssinfo) == 0)
		  {
		      (void) fprintf (stderr, "%s: XScreenSaverQueryInfo - "
				      "extension not supported\n", PROGNAME);
		      exit (EXIT_FAILURE);
		  }

		if (ssinfo->state != ScreenSaverOn)
		    return;
		else
		  {
		      (void) XSetErrorHandler (errorhandler);
		      (void) XCopyArea (display, buffer, ssinfo->window, gc,
					0, 0, 2, screen_height, (i - 1), 0);
		      (void) XSetErrorHandler (NULL);
		  }
		(void) XFlush (display);
		(void) usleep (5000);
		i--;
	    }
	  break;
      default:
#ifdef USE_DEBUG
	  (void) fprintf (stderr, "%s: warning: default case used in "
			  "clear_screen\n", PROGNAME);
#endif
	  i = 0;
	  (void) XDrawLine (display, buffer, bgc, 0, 0,
			    (int) screen_width, 0);
	  (void) XDrawLine (display, buffer, gc, 0, 1, (int) screen_width, 1);

	  while (i <= (int) screen_height)
	    {
		if (XScreenSaverQueryInfo (display, root_window, ssinfo) == 0)
		  {
		      (void) fprintf (stderr, "%s: XScreenSaverQueryInfo - "
				      "extension not supported\n", PROGNAME);
		      exit (EXIT_FAILURE);
		  }

		if (ssinfo->state != ScreenSaverOn)
		    return;
		else
		  {
		      (void) XSetErrorHandler (errorhandler);
		      (void) XCopyArea (display, buffer, ssinfo->window,
					gc, 0, 0, screen_width, 2, 0, i);
		      (void) XSetErrorHandler (NULL);
		  }
		(void) XFlush (display);
		(void) usleep (5000);
		i++;
	    }
      }

/* Pixmap should be cleared before we use it in the next cycle. */
    (void) XFillRectangle (display, buffer, bgc, 0, 0, screen_width,
			   screen_height);
}



/*
 * Function which choose random color.
 */
void
colorand (void)
{
    srandom ((unsigned long) time (NULL));

    switch (random () % 7)
      {
      case 0:
	  strcpy (user_color, "red");
	  break;
      case 1:
	  strcpy (user_color, "green");
	  break;
      case 2:
	  strcpy (user_color, "blue");
	  break;
      case 3:
	  strcpy (user_color, "yellow");
	  break;
      case 4:
	  strcpy (user_color, "magenta");
	  break;
      case 5:
	  strcpy (user_color, "cyan");
	  break;
      case 6:
	  strcpy (user_color, "white");
	  break;
      default:
	  strcpy (user_color, "yellow");
	  break;
      }
}



/*
 * Function that sets color.
 */
void
set_color (char *new_color)
{
    if (XParseColor (display, colormap, new_color, &color) == 0)
      {
	  fprintf (stderr, "%s: could not find color %s in server database\n",
		   PROGNAME, new_color);
	  exit (EXIT_FAILURE);
      }

    if (XAllocColor (display, colormap, &color) == 0)
      {
	  fprintf (stderr, "%s: could not allocate colors\n", PROGNAME);
	  exit (EXIT_FAILURE);
      }

    (void) XSetForeground (display, gc, color.pixel);
}



/*
 * Function used as error handler.
 */
int
errorhandler (Display * display, XErrorEvent * errorp)
{
#ifdef USE_DEBUG
    (void) fprintf (stderr, "%s: error occurred but ignored\n", PROGNAME);
#endif

    return ((int) errorp);
}



/*
 * Function used as signal handler.
 */
void
sighandler (int signum)
{
#ifdef USE_DEBUG
    (void) fprintf (stderr, "%s: signal %d catched\n", PROGNAME, signum);
#endif

/*
 * The original Unix, System V and libc4/5 systems would reset the handler
 * to SIG_DFL. Because of this, every time we catch signal we are interested
 * in, we are seting our handler for that signal again.
 * Client does not really need this because upon catching the signal it is
 * going to close connection to X server and exit. So, I will just comment
 * this line.
 */
/* signal (signum, sighandler); */

/* Removing _MIT_SCREEN_SAVER_ID property from root window */
    if (XScreenSaverUnregister (display, screen) == 0)
	(void) fprintf (stderr, "%s: XScreenSaverUnregister - unable to "
			"unregister\n", PROGNAME);

/*
 * We can now unset previously set screensaver attributes but I will leave
 * this line commented because of bug in Xfree86 and Xorg which would lead
 * to xserver crash if XScreenSaverUnsetAttributes() is called before
 * XCloseDisplay(). This bug is fixed in newer versions.
 */
/* (void) XScreenSaverUnsetAttributes (display, root_window); */

/* Closing display */
    (void) XCloseDisplay (display);

/* This will exit with 0 on most platforms */
    exit (EXIT_SUCCESS);
}
